<?php

/**
 * Check if either or both of ACS Points and BOX NOW Delivery are active
 * @return bool
 */
function loc_up_check_required_plugins() {
    $acs_points_active = in_array( "acs-points/acs-points-plugin.php", apply_filters( "active_plugins", get_option( "active_plugins" ) ) );
    $box_now_active = in_array( "box-now-delivery/box-now-delivery.php", apply_filters( "active_plugins", get_option( "active_plugins" ) ) );

    if( ! $acs_points_active && ! $box_now_active ) {
        return false;
    }
    return true;
}

/**
 * Output a notice on the admin dashboard
 * @param string $message
 * @param string $type
 * @return void
 */
function loc_up_admin_message( $class, $message ) {
    $class = strtolower( $class );
    switch( $class ) {
        case "success":
            $message_icon = "<span class='dashicons dashicons-yes-alt'></span> ";
            break;
        case "info":
            $message_icon = "<span class='dashicons dashicons-info'></span> ";
            break;
        case "error":
            $message_icon = "<span class='dashicons dashicons-warning'></span> ";
            break;
        default:
            $message_icon = "<span class='dashicons dashicons-info'></span> ";
    }
    $message = $message_icon . $message;

    add_action( "admin_notices", function() use( $class, $message ) {
        printf( "<div class='notice notice-%s is-dismissible'><p>%s</p></div>", $class, $message );
    } );
}

/**
 * Log to a txt file in /logs/
 * @param mixed $message
 * @return void
 */
function loc_up_log( $message = "" ) {
    $log_path = plugin_dir_path( __FILE__ ) . "logs/log.txt";
    $message = date( "d/m/Y H:i:s A" ) . " --- " . $message . "\n";
    file_put_contents( $log_path, $message, FILE_APPEND );
}

/**
 * Convert a measurement unit to points for DomPdf
 * @param float $value The numeric value
 * @param mixed $unit The unit. Expects: mm, cm, in. Defaults to mm
 * @throws \InvalidArgumentException
 * @return float The converted value in points (pt)
 */
function to_points( $value, $unit = "mm" ) {
    $unit = strtolower( $unit );
    switch( $unit ) {
        case "mm";
            return $value * 2.83465;
        case "cm";
            return $value * 28.3465;
        case "in";
            return $value * 72;
        default;
            throw new InvalidArgumentException( "Unsupported unit: $unit" );
    }
}

/**
 * Fetch and return ACS point metadata array
 * @param int $order_id
 * @return array acs_data with array_keys `id, name, address, postal_code`
 */
function loc_up_get_acs_meta( $order_id ) {
    $acs_data = array(
        "id" => null,
        "name" => null,
        "address" => null,
        "postal_code" => null
    );

    # Fetch the metadata
    $acs_point_data = get_post_meta( $order_id, "acs_pp_point", true );
    # Decode the string from json
    $acs_point_data_arr = json_decode( $acs_point_data );

    # Push new data to the $acs_data array
    if( ! empty( $acs_point_data_arr->name ) ) {
        $acs_data["name"] = esc_html( $acs_point_data_arr->name );
    }

    # Build the address using street, area and city
    $acs_address = "";
    if( ! empty( $acs_point_data_arr->street ) ) $acs_address .= $acs_point_data_arr->street;
    if( ! empty( $acs_point_data_arr->area ) ) $acs_address .= ", " . $acs_point_data_arr->area;
    if( ! empty( $acs_point_data_arr->city ) ) $acs_address .= ", " . $acs_point_data_arr->city;
    if( ! empty( $acs_address ) ){
        $acs_data["address"] = $acs_address;
    }

    if( ! empty( $acs_point_data_arr->sa_zipcode ) ){
        $acs_data["postal_code"] = esc_html( $acs_point_data_arr->sa_zipcode );
    }

    return $acs_data;
}

/**
 * Get Box Now pickup point address from order meta
 * @param object $order
 * @return array with keys `locker_id, name, address, postal_code`
 */
function loc_up_get_box_now_meta( $order ) {
    $boxnow_locker_id = $order->get_meta( "_boxnow_locker_id", true );
    $boxnow_locker_name = $order->get_meta( "_boxnow_locker_name", true );
    $boxnow_locker_address = $order->get_meta( "_boxnow_locker_address", true );
    $boxnow_locker_postal_code = $order->get_meta( "_boxnow_locker_postal_code", true );

    return array( 
        "locker_id" => $boxnow_locker_id, 
        "name" => $boxnow_locker_name, 
        "address" => $boxnow_locker_address, 
        "postal_code" => $boxnow_locker_postal_code 
    );
}

/**
 * Format pickup address as string for emails
 * @param array $address_data The $orders ACS Point or BOX Now meta data
 * @param string $service_name ACS Point or BOX Now
 * @return array custom shipping point address
 */
function loc_up_format_pickup_address_string( $address_data, $service_name ) {
    $custom_shipping_point = array(
                "first_name" => "",
                "last_name"  => "",
                "company"    => $service_name,
                "address_1"  => "",
                "address_2"  => "",
                "city"       => "",
                "state"      => "",
                "postcode"   => "",
                "country"    => "",
    );    
    
    if ( isset( $address_data["name"] ) ) {
        $custom_shipping_point["first_name"] = $address_data["name"];
    }
    
    if ( isset( $address_data["address"] ) ) {
        $custom_shipping_point["address_1"] = $address_data["address"];
    }
    
    if( isset( $address_data["postal_code"] ) || isset( $address_data["zip"] ) ) {
        $custom_shipping_point["postcode"] = $address_data["postal_code"] ?? $address_data["zip"];
    }

    return $custom_shipping_point;
}