<?php

/**
 * Plugin name: Locker Up
 * Plugin URL: https://mishimou-business.com/plugins/locker-up
 * Description: A plugin that works with the plugins: ACS Points Plugin & BOX NOW Delivery. Displays locker ID & location to customer AND creates a custom shipping label and packing slip
 * Version: 1.0.0
 * Requires at least: 5.2
 * Requires PHP: 8.0
 * Author: Mishimou Business
 * Author URI: https://mishimou-business.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: locker-up
 * Domain Path: /languages
 * Required Plugins: ACS Points, BOX NOW Delivery
 */

if(!defined("ABSPATH")) {
    exit;
}

# Define constants
define( "LOC_UP_PRIVATE_DIR", WP_CONTENT_DIR . "/locker_up_pdfs" ); # Define a private directory to store pdfs
define( "LOC_UP_PLUGIN_PATH", plugins_url( "", __FILE__ ) );

/**
 * Function ran when the plugin is installed
 * @return void
 */
function loc_up_plugin_installation() {
    # Create the private directory
    if( ! file_exists( LOC_UP_PRIVATE_DIR ) ) {
        wp_mkdir_p( LOC_UP_PRIVATE_DIR );
    }  
}
register_activation_hook( __FILE__, "loc_up_plugin_installation" );


/**
 * Register packing/shipping label settings
 * @return void
 */
function loc_up_register_plugin_settings() {
    register_setting( "loc_up_plugin_info_settings_group", "loc_up_license_key" );

    register_setting( "loc_up_settings_group", "packing_label_size", array( "default" => "A4" ) );
    register_setting( "loc_up_settings_group", "packing_logo" );
    register_setting( "loc_up_settings_group", "vat_number" );
    register_setting( "loc_up_settings_group", "greeting" );
    
    register_setting( "loc_up_settings_group", "shipping_label_size", array( "default" => "A4" ) );
    
    register_setting( "loc_up_settings_group", "packing_roll_width", array( "default" => "240" ) );
    
    register_setting( "loc_up_settings_group", "shipping_roll_width", array( "default" => "240" ) );
    
    register_setting( "loc_up_settings_group", "shipping_return_address", array( "default" => "" ) );
}

# Register settings
add_action( "admin_init", "loc_up_register_plugin_settings" );

# Required files
require_once "helpers.php";
require_once "locker-up-files.php";
require_once "admin/license.php";
require_once "admin/pages/admin.php";

/**
 * Enqueue plugin ADMIN CSS and JS
 * @return void
 */
function loc_up_enqueue_admin_scripts() {
    # Enqueue public styles
    $loc_up_admin_css_ver = date( "ymd-Gis", filemtime( plugin_dir_path( __FILE__ ) . "/admin/locker-up-admin.css" ) );
    wp_enqueue_style( "locker-up-admin", plugin_dir_url( __FILE__ ) . "/admin/locker-up-admin.css", array(), $loc_up_admin_css_ver );
    
    # Enqueue public scripts
    $loc_up_admin_js_ver = date( "ymd-Gis", filemtime( plugin_dir_path( __FILE__ ) . "/admin/locker-up-admin.js" ) );
    wp_enqueue_script( "locker-up-admin", plugin_dir_url( __FILE__ ) . "/admin/locker-up-admin.js", array( "jquery" ), $loc_up_admin_js_ver, true );
}
add_action('admin_enqueue_scripts', 'loc_up_enqueue_admin_scripts');

/**
 * Enqueue plugin PUBLIC CSS and JS
 * @return void
 */
function loc_up_enqueue_scripts() {
    # Enqueue public styles
    $loc_up_css_ver = date( "ymd-Gis", filemtime( dirname( __FILE__ ) . "/public/locker-up.css" ) );
    wp_enqueue_style( "locker-up", plugin_dir_url( __FILE__ ) . "/public/locker-up.css", array(), $loc_up_css_ver );
    
    # Enqueue public scripts
    $loc_up_js_ver = date( "ymd-Gis", filemtime( dirname( __FILE__ ) . "/public/locker-up.js" ) );
    wp_enqueue_script( "locker-up", plugin_dir_url( __FILE__ ) . "/public/locker-up.js", array( "jquery" ), $loc_up_js_ver, true );
}
add_action('wp_enqueue_scripts', 'loc_up_enqueue_scripts');


/**
 * Update order meta on checkout
 * @param object $order_id
 * @return void
 */
function loc_up_checkout_meta($order_id) {
    $order = wc_get_order($order_id);

    // ✅ Box Now Data Store (Already Working)
    if( ! empty( $_POST["box_now_selected_locker"] ) ) {
        $locker_data = json_decode( stripslashes($_POST["box_now_selected_locker"] ), true );
        if ($locker_data) {
            $order->update_meta_data( "_boxnow_locker_id", $locker_data["boxnowLockerId"] );
            $order->update_meta_data( "_boxnow_locker_name", $locker_data["boxnowLockerName"] );
            $order->update_meta_data( "_boxnow_locker_address", $locker_data["boxnowLockerAddressLine1"] );
            $order->update_meta_data( "_boxnow_locker_postal_code", $locker_data["boxnowLockerPostalCode"] );
        }
    }

    // ✅ ACS Points Data Store
    if ( ! empty( $_POST["acs_pp_point_id"] ) ) {
        $order->update_meta_data( "_acs_point_id", sanitize_text_field( $_POST["acs_pp_point_id"] ) );
        loc_up_log("✅ ACS Point Stored:" . $_POST["acs_pp_point_id"] );
    } else {
        loc_up_log( "❌ ACS Point Not Found" );
    }

    // ✅ Save Order Meta
    $order->save();
}
add_action( "woocommerce_checkout_update_order_meta", "loc_up_checkout_meta" );

/**
 * Show data in Order Details Admin Page
 * @param object $order
 * @return void
 */
function loc_up_orders_page ( $order ) {
    $order_id = $order->get_id();

    if (!empty($acs_point_id) || !empty($boxnow_locker_id)) {
        echo "<h3>📦 Delivery Details</h3>";
    }

    # ACS Point data
    $acs_point_id = $order->get_meta( "_acs_point_id", true );
    if ( ! empty( $acs_point_id ) ) {
        $acs_point_data = loc_up_get_acs_meta( $order_id );

        echo "<h4>📍 ACS Point</h4>";
        echo null == ( $acs_point_data["id"] ) ? "" :  '<p><strong>🔹 ACS Point ID:</strong> ' . $acs_point_data["id"] . '</p>';
        echo null == ( $acs_point_data["name"] ) ? "" :  '<p><strong>🔹 Name:</strong> ' .$acs_point_data["name"] . '</p>';
        echo null == ( $acs_point_data["address"] ) ? "" :  '<p><strong>🔹 Address:</strong> ' . $acs_point_data["address"] . '</p>';
        echo null == ( $acs_point_data["postal_code"] ) ? "" :  '<p><strong>🔹 Postal Code:</strong> ' . $acs_point_data["postal_code"] . '</p>';
    }
    
    # Box Now data
    $boxnow_locker_id = $order->get_meta('_boxnow_locker_id', true);
    $box_now_data = loc_up_get_box_now_meta( $order );
    if ( ! empty( $boxnow_locker_id ) ) {
        echo '<h4>📍 Box Now Locker</h4>';
        echo '<p><strong>🔹 Locker ID:</strong> ' . esc_html( $box_now_data["locker_id"] ) . '</p>';
        echo '<p><strong>🔹 Name:</strong> ' . esc_html( $box_now_data["name"] ) . '</p>';
        echo '<p><strong>🔹 Address:</strong> ' . esc_html( $box_now_data["address"] ) . '</p>';
        echo '<p><strong>🔹 Postal Code:</strong> ' . esc_html( $box_now_data["postal_code"] ) . '</p>';
    }

    # Show links to the packaging and shipping pdfs
    # Attach a nonce to verify downloads
    $key = "f0c91a4dcc";

    echo "🗂️ Download the <strong><a href='" . home_url( "/wp-json/lockerup-pdfs/v1/download?file=packing_slip_order_$order_id.pdf&order_id=$order_id&key=$key" ). "' target='_blank'>Packaging slip here</a></strong> <br>";
    echo "🗂️  Download the <strong><a href='" . home_url( "/wp-json/lockerup-pdfs/v1/download?file=shippinglabel_order_$order_id.pdf&order_id=$order_id&key=$key" ). "' target='_blank'>Shipping label here</a></strong> <br>";
}
add_action( "woocommerce_admin_order_data_after_billing_address", "loc_up_orders_page", 10, 1);

/**
 * Modify shipping address display on the checkout page
 * @param array $address
 * @param object $order
 * 
 * @return array address
 */
function loc_up_modify_checkout_shipping_address( $address, $order ) {
    if( ! $order ) return $address;
    $order_id = $order->get_id();

    $acs_point_id = $order->get_meta( "_acs_point_id", true );
    # ACS Point data
    if ( ! empty( $acs_point_id ) ) {
        $acs_point_data = loc_up_get_acs_meta( $order_id );
        return loc_up_format_pickup_address_string( $acs_point_data, "ACS Point Pickup Point" );
    }

    $boxnow_locker_id = $order->get_meta( "_boxnow_locker_id", true );
    # BOX Now data
    if( ! empty( $boxnow_locker_id ) ) {
        $box_now_data = loc_up_get_box_now_meta( $order );
        return loc_up_format_pickup_address_string( $box_now_data, "BOX Now Pickup Point" ); 
    }
    
    return $address; # Default shipping address
}
add_filter( "woocommerce_order_formatted_shipping_address", "loc_up_modify_checkout_shipping_address", 10,2 );

if( ! class_exists( "\Dompdf\Dompdf" ) ) { # only require if it doesn't already exist (could've been added by another plugin)
    require_once plugin_dir_path( __FILE__ ) . "vendor/autoload.php";
}

use Dompdf\Dompdf;
use Dompdf\Options;
/**
 * Create packing and shipping labels
 * @param array $attachments
 * @param string $email_id
 * @param object $order
 * @return array attachments
 */
function loc_up_attach_pdfs_to_email($attachments, $email_id, $order) {
    # Check if either BOX Now or ACS Points or BOTH are active
    if( ! loc_up_check_required_plugins() ) {
        return $attachments;
    }

    if ( !is_a($order, 'WC_Order') ) {
        return $attachments;
    }
    
	// $admin_emails = array( "new_order", "cancelled_order", "failed_order", "customer_completed_order", "customer_processing_order" );
    if( $email_id !== "new_order" ) { # Only attach to admin email
        return $attachments;
    }
    
    try {
        $order_id = $order->get_id();
        $order_date = $order->get_date_created()->date('Y-m-d H:i:s');
        $billing_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
        $billing_email = $order->get_billing_email();
        $subtotal = $order->get_subtotal();
        $shipping_total = $order->get_shipping_total();
        $payment_method = $order->get_payment_method_title();
        $total = $order->get_total();
        $shipping_method = $order->get_shipping_method();
        $boxnow_locker_id = $order->get_meta('_boxnow_locker_id', true);
        $boxnow_locker_name = $order->get_meta('_boxnow_locker_name', true);
        $boxnow_locker_address = $order->get_meta('_boxnow_locker_address', true);
        $boxnow_locker_postal_code = $order->get_meta('_boxnow_locker_postal_code', true);
        $acs_point_id = $order->get_meta('_acs_point_id', true);
        $warehouse_id = $order->get_meta('_selected_warehouse', true);
        $shipping_address = $order->get_formatted_shipping_address();
        
        # Generate Packing Slip PDF
    
        # Fetch paper dimensions from the options table
        $loc_up_packing_logo = get_option( "packing_logo", "" );
        $loc_up_greeting = get_option( "greeting", "" );
    
        $html_packing_slip = '<!DOCTYPE html>
        <html>
        <head>
            <style>
                body { font-family: DejaVu Sans, sans-serif; font-size: 10px; margin: 0; padding: 10px 50px;  }
                .container { padding: 1px; }
                table { margin-left: auto; margin-right: auto; }
                .title { font-size: 10px; font-weight: bold; text-align: center; }
                .order-info p { margin: 4px 0; }
                .order-details { width: 100%; border-collapse: collapse; }
                .order-details th, .order-details td { border: 1px solid #ddd; padding: 2px; }
                .order-details th { background: black; color: white; text-align: center; }
                html {margin: 1px; }
            </style>
        </head>
        <body>
            <div class="container">
                <img src="' . $loc_up_packing_logo . '" alt="Shop logo" width="100px" />
                <div class="title">Packing Slip & Receipt</div>
                <div class="order-info">
                    <p><strong>Order ID:</strong> #' . $order_id . '</p>
                    <p><strong>Order Date:</strong> ' . esc_html($order_date) . '</p>
                    <p><strong>Customer:</strong> ' . esc_html($billing_name) . '</p>
                    <p><strong>Email:</strong> ' . esc_html($billing_email) . '</p>
                </div>
    
                <table class="order-details">
                    <tr>
                        <th>Product</th>
                        <th>Qty</th>
                        <th>Price</th>
                    </tr>';
    
        foreach ($order->get_items() as $item) {
            $product_name = $item->get_name();
            $quantity = $item->get_quantity();
            $price = wc_price($item->get_total());
            $html_packing_slip .= '<tr>
                        <td>' . esc_html($product_name) . '</td>
                        <td>' . esc_html($quantity) . '</td>
                        <td>' . $price . '</td>
                    </tr>';
        }
    
        $html_packing_slip .= '<tr><td colspan="2"><strong>Subtotal:</strong></td><td>' . wc_price($subtotal) . '</td></tr>';
        $html_packing_slip .= '<tr><td colspan="2"><strong>Shipping:</strong></td><td>' . wc_price($shipping_total) . '</td></tr>';
        if (!empty($acs_point_id) || !empty($boxnow_locker_id)) {
            if (!empty($boxnow_locker_id)) {
                $html_packing_slip .= '<tr><td colspan="3"><strong>Boxnow Locker ID:</strong> ' . esc_html($boxnow_locker_id) . '</td></tr>';
            }
            if (!empty($boxnow_locker_name)) {
                $html_packing_slip .= '<tr><td colspan="3"><strong>Boxnow Locker Name:</strong> ' . esc_html($boxnow_locker_name) . '</td></tr>';
            }
            if (!empty($boxnow_locker_address)) {
                $html_packing_slip .= '<tr><td colspan="3"><strong>Boxnow Locker Name:</strong> ' . esc_html($boxnow_locker_address) . '</td></tr>';
            }
            if (!empty($boxnow_locker_postal_code)) {
                $html_packing_slip .= '<tr><td colspan="3"><strong>Boxnow Locker Name:</strong> ' . esc_html($boxnow_locker_postal_code) . '</td></tr>';
            }
            if (!empty($acs_point_id)) {
                $html_packing_slip .= '<tr><td colspan="3"><strong>ACS Point ID:</strong> ' . esc_html($acs_point_id) . '</td></tr>';
            }
        } else {
            $html_packing_slip .= '<tr><td colspan="3"><strong>Shipping Address:</strong> ' . wp_kses_post($shipping_address) . '</td></tr>';
        }
        
        $html_packing_slip .= '<tr><td colspan="2"><strong>Payment Method:</strong></td><td>' . esc_html($payment_method) . '</td></tr>';
        
        // Group tax amounts by rate label
        $vat_breakdown = [];
    
        foreach ( $order->get_items( 'tax' ) as $tax_item ) {
            $tax_total = $tax_item->get_tax_total();
            $tax_label = $tax_item->get_label();
    
            if ( $tax_total > 0 ) {
                if ( ! isset( $vat_breakdown[ $tax_label ] ) ) {
                    $vat_breakdown[ $tax_label ] = 0;
                }
                $vat_breakdown[ $tax_label ] += $tax_total;
            }
        }
    
        // Build the VAT breakdown text
        $vat_text_parts = [];
        foreach ( $vat_breakdown as $label => $amount ) {
            $vat_text_parts[] = wc_price( $amount ) . ' ' . esc_html( $label );
        }
    
        $vat_summary = ! empty( $vat_text_parts ) ? ' (includes ' . implode( ', ', $vat_text_parts ) . ')' : '';
        $html_packing_slip .= '<tr><td colspan="2"><strong>Total:</strong></td><td>' . wc_price( $total ) . $vat_summary . '</td></tr>';
         
        $html_packing_slip .= '</table>
            <div class="footer">' . $loc_up_greeting .'</div>
            </div>
        </body>
        </html>';
        
    
        # Create a new pdf
        $packing_slip_dompdf_options = new Options();
        $packing_slip_dompdf_options->set( "isRemoteEnabled", true );
        $packing_slip_dompdf = new Dompdf( $packing_slip_dompdf_options );
        $packing_slip_dompdf->loadHtml($html_packing_slip);
        $packing_slip_saved_width = get_option( "packing_roll_width" ); # get packing_roll_width from the database
        $packing_slip_width = to_points( intval( $packing_slip_saved_width ), "mm" ); # convert it to points (pt)
        $packing_slip_height = 0;
    
        # Get pdf body height by setting a callback
        $packing_slip_dompdf->setCallbacks( array(
            "myCallbacks" => array(
                "event" => "end_frame",
                "f" => function( $frame ) use( &$packing_slip_height ) {
                    $node = $frame->get_node();
                    if( strtolower( $node->nodeName ) === "body" ) {
                        $padding_box = $frame->get_padding_box();
                        $packing_slip_height = $padding_box["h"] + 50; # add 50pts to the height 
                    }
                }
            )
        ) );
        $packing_slip_dompdf->render();
        # Destroy the old pdf
        unset( $packing_slip_dompdf );
            
        #Create a new pdf again since we have the height we need
        $packing_slip_dompdf_options = new Options();
        $packing_slip_dompdf_options->set( "isRemoteEnabled", true );
        $packing_slip_dompdf = new Dompdf( $packing_slip_dompdf_options );
        $packing_slip_dompdf->loadHtml( $html_packing_slip );
        $loc_up_packing_label_size = get_option( "packing_label_size", "" ); # A4|Paper roll
        if( $loc_up_packing_label_size == "A4" ) { # check if saved packing_label_size is A4
            $packing_slip_dompdf->setPaper( "A4", "potrait" ); 
        } else {
            $packing_slip_dompdf->setPaper(array( 0, 0, $packing_slip_width, $packing_slip_height ) ); 
        }
        $packing_slip_dompdf->render();
    
        # Store the pdf
        $packing_slip_pdf_path = LOC_UP_PRIVATE_DIR . '/packing_slip_order_' . $order_id . '.pdf';
        file_put_contents( $packing_slip_pdf_path, $packing_slip_dompdf->output() );
        $attachments[] = $packing_slip_pdf_path;
    
        loc_up_log( "Packing slip PDF created in $packing_slip_pdf_path" );
    
        # Generate Shipping Label PDF
        $loc_up_shipping_label_size = get_option( "shipping_label_size", "" );
        if( $loc_up_shipping_label_size !== "A4" ) { # only if paper size is not A4
            $shipping_label_css = "<style>
                html, body { height;100%; }
                body { font-family: DejaVu Sans, sans-serif; font-size: 14px; margin: 0; padding: 0; }
                table { 
                    position: absolute;
                    top: 50%;
                    left: 0px;
                    transform: translate(0, -50%);
                    border: 1px solid #000;
                    border-collapse: collapse; 
                    page-break-before: avoid; 
                    margin-left: 8px; 
                }
                td, th { border: 1px solid #000; padding: 12px 4px; max-width:100%; white-space:nowrap; page-break-before: avoid; page-break-inside: avoid; page-break-after: avoid; vertical-align: top;}
                html {margin: 1px; }
            </style>";
        } else {
            $shipping_label_css = "<style>
                body { font-family: DejaVu Sans, sans-serif; font-size: 14px; margin: 0; padding: 0;  }
                td, th { border: 1px solid #000; padding: 12px 4px; max-width:100%; white-space:nowrap; page-break-before: avoid; page-break-inside: avoid; page-break-after: avoid; vertical-align: top;}
            </style>";
        }

        $html_shipping_label = '<!DOCTYPE html>
        <html>
        <head> ' . $shipping_label_css . ' </head>
        <body>
            <div style="position:relative;">
                <table>
                    <tr>
                        <td>
                        <strong>DELIVERY ADDRESS</strong><br>
                        Order: #' . $order_id . '<br>
                        Name: ' . esc_html($billing_name) . '<br>';
        
                        if ( ! empty($boxnow_locker_id ) ) {
                            $html_shipping_label .= 'Locker ID: ' . esc_html($boxnow_locker_id) . ' BOXNOW<br>
                                Warehouse ID: ' . esc_html($warehouse_id) . '<br>
                                Locker Name: ' . esc_html($boxnow_locker_name) . '<br>
                                Locker Address: ' . esc_html($boxnow_locker_address) . '<br>
                                Postal Code: ' . esc_html($boxnow_locker_postal_code) . '<br>';
                        } elseif ( ! empty($acs_point_id ) ) {
                            $html_shipping_label .= 'ACS Point ID: ' . esc_html($acs_point_id) . '<br>';
                            $acs_point_data = loc_up_get_acs_meta( $order_id );
        
                            if( null !== ( $acs_point_data["name"] ) ) $html_shipping_label .= 'ACS Point Name: ' . $acs_point_data["name"] . '<br>';
                            if( null !== ( $acs_point_data["address"] ) ) $html_shipping_label .= 'Address: ' . $acs_point_data["address"] . '<br>';
                            if( null !== ( $acs_point_data["postal_code"] ) ) $html_shipping_label .= 'Postal Code: ' . $acs_point_data["postal_code"] . '<br>';
                        } else {
                            $html_shipping_label .= 'Shipping Address: ' . wp_kses_post($shipping_address) . '<br>';
                        }
        
                        $html_shipping_label .= '</td>
                        <td>
                            <strong>RETURN ADDRESS</strong><br>
                            <br>' . esc_textarea( get_option( "shipping_return_address", "" ) ) . '
                        </td>
                    </tr>
                </table>
            </div>
        </body>
        </html>';
    
        # Create a new shipping_label pdf
        $shipping_label_dompdf = new Dompdf();
        $shipping_label_dompdf->loadHtml( $html_shipping_label );
        $shipping_label_saved_width = get_option( "shipping_roll_width" ); # get shipping_roll_width from the database
        $shipping_label_width = 0; # because it's landscape, interchange the width & height
        $shipping_label_height = to_points( intval( $shipping_label_saved_width ), "mm" ); # because it's landscape, interchange the width & height
    
        # Get pdf body width by setting a callback
        $shipping_label_dompdf->setCallbacks( array(
            "myCallbacks" => array(
                "event" => "end_frame",
                "f" => function( $frame ) use( &$shipping_label_width ) {
                    $node = $frame->get_node();
                    if( strtolower( $node->nodeName ) === "body" ) {
                        $padding_box = $frame->get_padding_box();
                        $shipping_label_width = $padding_box["w"];
                    }
                }
            )
        ) );

        $shipping_label_dompdf->render();
        # Destroy the old shipping_label pdf
        unset( $shipping_label_dompdf );
            
        #Create a new shipping_label pdf again since we have the width we need
        $shipping_label_dompdf = new Dompdf();
        $shipping_label_dompdf->loadHtml($html_shipping_label );
    
        if( $loc_up_shipping_label_size == "A4" ) { # check if saved shipping_label_size is A4
            $shipping_label_dompdf->setPaper( "A4", "landscape" ); 
        } else {
            $shipping_label_dompdf->setPaper(array( 0, 0, $shipping_label_width, $shipping_label_height ) ); 
        }
        $shipping_label_dompdf->render();
        
        # Store the pdf
        $shipping_label_pdf_path = LOC_UP_PRIVATE_DIR . '/shippinglabel_order_' . $order_id . '.pdf';
    
        loc_up_log( "Packing slip PDF created in $shipping_label_pdf_path" );
    
        file_put_contents($shipping_label_pdf_path, $shipping_label_dompdf->output() );
        $attachments[] = $shipping_label_pdf_path;
        return $attachments;
    } catch( Exception $e ) {
        error_log( "EMAIL ATTACHMENTS ERROR: " . $e->getMessage() );
        return $attachments;
    }
}

add_action( "woocommerce_email_attachments", "loc_up_attach_pdfs_to_email", 10, 3 );